# Copyright (C) 2001-2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: rdataclass.py,v 1.8 2003/06/24 02:37:00 halley Exp $

"""DNS Rdata Classes.

@var _by_text: The rdata class textual name to value mapping
@type _by_text: dict
@var _by_value: The rdata class value to textual name mapping
@type _by_value: dict
@var metaclasses: If an rdataclass is a metaclass, there will be a mapping
whose key is the rdatatype value and whose value is True in this dictionary.
@type metaclasses: dict"""

import re

import dns.exception

RESERVED0 = 0
IN = 1
CH = 3
HS = 4
NONE = 254
ANY = 255

_by_text = {
    'RESERVED0' : RESERVED0,
    'IN' : IN,
    'INTERNET' : IN,
    'CH' : CH,
    'CHAOS' : CH,
    'HS' : HS,
    'HESIOD' : HS,
    'NONE' : NONE,
    'ANY' : ANY
    }

_by_value = {
    RESERVED0 : 'RESERVED0',
    IN : 'IN',
    CH : 'CH',
    HS : 'HS',
    NONE : 'NONE',
    ANY : 'ANY'
    }

metaclasses = {
    NONE : True,
    ANY : True
    }

_unknown_class_pattern = re.compile('CLASS([0-9]+)$', re.I);

class UnknownRdataclass(dns.exception.DNSException):
    """Raised when a class is unknown."""
    pass

def from_text(text):
    """Convert text into a DNS rdata class value.
    @rtype: int"""
    
    value = _by_text.get(text.upper())
    if value is None:
        match = _unknown_class_pattern.match(text)
        if match == None:
            raise UnknownRdataclass
        value = int(match.group(1))
    return value

def to_text(value):
    """Convert a DNS rdata class to text.
    @rtype: string"""
    
    if value < 0 or value > 65535:
        raise UnknownRdataclass
    text = _by_value.get(value)
    if text is None:
        text = 'CLASS' + `value`
    return text

def is_metaclass(rdclass):
    """True if the class is a metaclass.
    @rtype: bool"""
    
    if metaclasses.has_key(rdclass):
        return True
    return False
