# Copyright (C) 2001-2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: node.py,v 1.7 2003/06/24 02:37:00 halley Exp $

"""DNS nodes.  A node is a named set of rdatasets."""

import StringIO

import dns.rdataset
import dns.rdatatype

class Node(object):
    """A DNS node.
    
    A node is a named set of rdatasets

    @ivar name: the node's name
    @type name: dns.name.Name object
    @ivar rdatasets: the node's rdatasets
    @type rdatasets: list of dns.rdataset.Rdataset objects"""
    
    def __init__(self, name):
        """Initialize a DNS node.

        @param name: The node's name
        @type name: dns.name.Name object."""
        
        self.name = name
        #
        # We use a list instead of a set so we can do the things (i.e.
        # having duplicate rdataset types) that dynamic update needs.
        #
        self.rdatasets = [];

    def to_text(self, **kw):
        """Convert a node to text format.

        Each rdataset at the node is printed.  Any keyword arguments
        to this method are passed on to the rdataset's to_text() method.
        @rtype: string"""
        
        s = StringIO.StringIO()
        for rds in self.rdatasets:
            print >> s, rds.to_text(self.name, **kw)
        return s.getvalue()[:-1]

    def __repr__(self):
        return '<DNS node ' + str(self.name) + '>'
    
    def __str__(self):
        return self.to_text()
    
    def __eq__(self, other):
        """Two nodes are equal if they have the same name and have the
        same rdatasets.
        @rtype: bool"""
        if self.name != other.name:
            return False
        #
        # This is inefficient.  Good thing we don't need to do it much.
        #
        for rd in self.rdatasets:
            if rd not in other.rdatasets:
                return False
        for rd in other.rdatasets:
            if rd not in self.rdatasets:
                return False
        return True

    def __ne__(self, other):
        return not self.__eq__(other)
        
    def __len__(self):
        return len(self.rdatasets)

    def __iter__(self):
        return iter(self.rdatasets)

    def find_rdataset(self, rdclass, rdtype, covers=dns.rdatatype.NONE,
                      create=False, force_unique=False, deleting=0):
        """Find an rdataset matching the specified properties in the
        current node.

        @param rdclass: The class of the rdataset
        @type rdclass: int
        @param rdtype: The type of the rdataset
        @type rdtype: int
        @param covers: The covered type.  Usually this value is
        dns.rdatatype.NONE, but if the rdtype is dns.rdatatype.SIG,
        then the covers value will be the rdata type the SIG covers.
        The library treats the SIG type as if it were a family of
        types, e.g. SIG(A), SIG(NS), SIG(SOA).  This makes SIGs much
        easier to work with than if SIGs covering different rdata
        types were aggregated into a single SIG rdataset.
        @type covers: int
        @param create: If True, create the rdataset if it is not found, unless
        I{force_unique} is also True, in which case always create a new
        rdataset.  The created rdataset is appended to self.rdatasets.
        @type create: bool
        @param force_unique: If I{create} is True, always create the rdataset,
        even if it already exists.
        @type force_unique: bool
        @param deleting: If non-zero, the value of deleting should be
        the class to use when converting the rdataset to text or wire
        format.  This field is used in dynamic update operations, for
        example in the "Delete an RR from an RRset" case, the deleting
        value will be dns.rdatatype.NONE.
        @type deleting: int
        @rtype: dns.rdataset.Rdataset object"""

        if create and force_unique:
            rds = dns.rdataset.Rdataset(rdclass, rdtype, deleting)
            self.rdatasets.append(rds)
            return rds
        for rds in self.rdatasets:
            if rds.match(rdclass, rdtype, covers, deleting):
                return rds
        if not create:
            raise KeyError
        rds = dns.rdataset.Rdataset(rdclass, rdtype, deleting)
        self.rdatasets.append(rds)
        return rds

    def to_wire(self, file, compress=None, origin=None, question=False):
        """Return a string containing the node in DNS compressed wire
        format.

        @param file: The file to which the wire format data will be appended
        @type file: file
        @param compress: The compression table to use; the default is None.
        @type compress: dict
        @param origin: The origin to be appended to any relative names when
        they are emitted.  The default is None.
        @param question: If True, render this rdataset in the format of
        the question section, i.e. emit only the owner name, rdata class,
        and rdata type.
        @type question: bool
        @rtype: string"""

        count = 0
        for rds in self.rdatasets:
            count += rds.to_wire(self.name, file, compress, origin, question)
        return count
