# Copyright (C) 2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: name.py,v 1.4 2003/06/24 02:37:00 halley Exp $

import unittest

import dns.name

class NameTestCase(unittest.TestCase):
    def setUp(self):
        self.origin = dns.name.from_text('example.')
        
    def testFromTextRel1(self):
        n = dns.name.from_text('foo.bar')
        self.failUnless(n.labels == ('foo', 'bar', ''))

    def testFromTextRel2(self):
        n = dns.name.from_text('foo.bar', origin=self.origin)
        self.failUnless(n.labels == ('foo', 'bar', 'example', ''))

    def testFromTextRel3(self):
        n = dns.name.from_text('foo.bar', origin=None)
        self.failUnless(n.labels == ('foo', 'bar'))

    def testFromTextAbs1(self):
        n = dns.name.from_text('foo.bar.')
        self.failUnless(n.labels == ('foo', 'bar', ''))

    def testImmutable1(self):
        def bad():
            self.origin.labels = ()
        self.failUnlessRaises(TypeError, bad)

    def testImmutable2(self):
        def bad():
            self.origin.labels[0] = 'foo'
        self.failUnlessRaises(TypeError, bad)

    def testAbs1(self):
        self.failUnless(dns.name.root.is_absolute())

    def testAbs2(self):
        self.failUnless(not dns.name.empty.is_absolute())

    def testAbs3(self):
        self.failUnless(self.origin.is_absolute())

    def testAbs3(self):
        n = dns.name.from_text('foo', origin=None)
        self.failUnless(not n.is_absolute())

    def testWild1(self):
        n = dns.name.from_text('*.foo', origin=None)
        self.failUnless(n.is_wild())

    def testWild2(self):
        n = dns.name.from_text('*a.foo', origin=None)
        self.failUnless(not n.is_wild())

    def testWild3(self):
        n = dns.name.from_text('a.*.foo', origin=None)
        self.failUnless(not n.is_wild())

    def testWild4(self):
        self.failUnless(not dns.name.root.is_wild())

    def testWild5(self):
        self.failUnless(not dns.name.empty.is_wild())

    def testHash1(self):
        n1 = dns.name.from_text('fOo.COM')
        n2 = dns.name.from_text('foo.com')
        self.failUnless(hash(n1) == hash(n2))

    def testCompare1(self):
        n1 = dns.name.from_text('a')
        n2 = dns.name.from_text('b')
        self.failUnless(n1 < n2)
        self.failUnless(n2 > n1)

    def testCompare2(self):
        n1 = dns.name.from_text('')
        n2 = dns.name.from_text('b')
        self.failUnless(n1 < n2)
        self.failUnless(n2 > n1)

    def testCompare3(self):
        self.failUnless(dns.name.empty < dns.name.root)
        self.failUnless(dns.name.root > dns.name.empty)

    def testSubdomain1(self):
        self.failUnless(not dns.name.empty.is_subdomain(dns.name.root))

    def testSubdomain2(self):
        self.failUnless(not dns.name.root.is_subdomain(dns.name.empty))

    def testSubdomain3(self):
        n = dns.name.from_text('foo', origin=self.origin)
        self.failUnless(n.is_subdomain(self.origin))

    def testSubdomain4(self):
        n = dns.name.from_text('foo', origin=self.origin)
        self.failUnless(n.is_subdomain(dns.name.root))

    def testSubdomain5(self):
        n = dns.name.from_text('foo', origin=self.origin)
        self.failUnless(n.is_subdomain(n))

    def testSuperdomain1(self):
        self.failUnless(not dns.name.empty.is_superdomain(dns.name.root))

    def testSuperdomain2(self):
        self.failUnless(not dns.name.root.is_superdomain(dns.name.empty))

    def testSuperdomain3(self):
        n = dns.name.from_text('foo', origin=self.origin)
        self.failUnless(self.origin.is_superdomain(n))

    def testSuperdomain4(self):
        n = dns.name.from_text('foo', origin=self.origin)
        self.failUnless(dns.name.root.is_superdomain(n))

    def testSuperdomain5(self):
        n = dns.name.from_text('foo', origin=self.origin)
        self.failUnless(n.is_superdomain(n))

    def testCanonicalize1(self):
        n = dns.name.from_text('FOO.bar', origin=self.origin)
        c = n.canonicalize()
        self.failUnless(c.labels == ('foo', 'bar', 'example', ''))

    def testToText1(self):
        n = dns.name.from_text('FOO.bar', origin=self.origin)
        t = n.to_text()
        self.failUnless(t == 'FOO.bar.example.')

    def testToText2(self):
        n = dns.name.from_text('FOO.bar', origin=self.origin)
        t = n.to_text(True)
        self.failUnless(t == 'FOO.bar.example')

    def testToText3(self):
        n = dns.name.from_text('FOO.bar', origin=None)
        t = n.to_text()
        self.failUnless(t == 'FOO.bar')

    def testToText4(self):
        t = dns.name.empty.to_text()
        self.failUnless(t == '@')

    def testToText5(self):
        t = dns.name.root.to_text()
        self.failUnless(t == '.')

    def testToText6(self):
        n = dns.name.from_text('FOO bar', origin=None)
        t = n.to_text()
        self.failUnless(t == r'FOO\032bar')

    def testToText7(self):
        n = dns.name.from_text(r'FOO\.bar', origin=None)
        t = n.to_text()
        self.failUnless(t == r'FOO\.bar')

    def testToText8(self):
        n = dns.name.from_text(r'\070OO\.bar', origin=None)
        t = n.to_text()
        self.failUnless(t == r'FOO\.bar')

if __name__ == '__main__':
    unittest.main()
