# Copyright (C) 2001-2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: flags.py,v 1.3 2003/06/06 02:31:57 halley Exp $

"""DNS Message Flags."""

QR = 0x8000
AA = 0x0400
TC = 0x0200
RD = 0x0100
RA = 0x0080
AD = 0x0020
CD = 0x0010

DO = 0x8000

_by_text = {
    'QR' : QR,
    'AA' : AA,
    'TC' : TC,
    'RD' : RD,
    'RA' : RA,
    'AD' : AD,
    'CD' : CD
}

_by_value = {
    QR : 'QR',
    AA : 'AA',
    TC : 'TC',
    RD : 'RD',
    RA : 'RA',
    AD : 'AD',
    CD : 'CD'
}

_edns_by_text = {
    'DO' : DO
}

_edns_by_value = {
    DO : 'DO'
}

def from_text(text):
    """Convert a space-separated list of flag text values into a flags
    value.
    @rtype: int"""
    
    flags = 0
    tokens = text.split()
    for t in tokens:
        flags = flags | _by_text[t.upper()]
    return flags

def to_text(flags):
    """Convert a flags value into a space-separated list of flag text
    values.
    @rtype: string"""
    
    s = ''
    pad = ''
    for k in _by_value.keys():
        if flags & k != 0:
            s = s + pad + _by_value[k]
            pad = ' '
    return s

def edns_from_text(text):
    """Convert a space-separated list of EDNS flag text values into a EDNS
    flags value.
    @rtype: int"""
    
    flags = 0
    tokens = text.split()
    for t in tokens:
        flags = flags | _edns_by_text[t.upper()]
    return flags

def edns_to_text(flags):
    """Convert an EDNS flags value into a space-separated list of EDNS flag
    text values.
    @rtype: string"""
    
    s = ''
    pad = ''
    for k in _edns_by_value.keys():
        if flags & k != 0:
            s = s + pad + _edns_by_value[k]
            pad = ' '
    return s
