# Copyright (C) 2001-2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: rdata.py,v 1.6 2003/06/06 02:17:15 halley Exp $

"""DNS rdata.

@var _rdata_modules: A dictionary mapping a (rdclass, rdtype) tuple to
the module which implements that type.
@type _rdata_modules: dict
@var _module_prefix: The prefix to use when forming modules names.  The
default is 'DNS.rdtypes'.  Changing this value will break the library.
@type _module_prefix: string
@var _hex_chunk: At most this many octets that will be represented in each
chunk of hexstring that _hexify() produces before whitespace occurs.
@type _hex_chunk: int"""

import codecs
import cStringIO

import DNS.rdataclass
import DNS.rdatatype
import DNS.tokenizer

class MeaninglessComparison(Exception):
    """Raised if an attemped is made to compare one rdata type with another."""
    pass

_hex_chunk = 32

def _hexify(data):
    hex = data.encode('hex_codec')
    l = len(hex)
    if l > _hex_chunk:
        chunks = []
        i = 0
        while i < l:
            chunks.append(hex[i : i + _hex_chunk])
            i += _hex_chunk
        hex = ' '.join(chunks)
    return hex

class Rdata(object):
    """Base class for all DNS rdata types.
    """
    
    def __init__(self, rdclass, rdtype):
        """Initialize an rdata.
        @param rdclass: The rdata class
        @type rdclass: int
        @param rdtype: The rdata type
        @type rdtype: int"""

	self.rdclass = rdclass
	self.rdtype = rdtype

    def covers(self):
        """DNS SIG rdatas apply to specific type; this type is
        returned by the covers() function.  If the rdata type is not
        SIG, DNS.rdatatype.NONE is returned.  This is useful when
        creating rdatasets, allowing the rdataset to contain only SIGs
        of a particular type, e.g. SIG(NS).
        @rtype: int"""
        
	return DNS.rdatatype.NONE

    def extended_rdatatype(self):
        """Return a 32-bit type value, the least significant 16 bits of
        which are the ordinary DNS type, and the upper 16 bits of which are
        the "covered" type, if any.
        @rtype: int"""
        return self.covers() << 16 | self.rdtype

    def to_text(self, **kw):
        """Convert an rdata to text format.
        @rtype: string"""
        raise NotImplementedError        

    def to_wire(self, file, compress = None, origin = None):
        """Convert an rdata to wire format.
        @rtype: string"""
        raise NotImplementedError
        
    def __repr__(self):
	covers = self.covers()
        if covers == DNS.rdatatype.NONE:
            ctext = ''
        else:
            ctext = '(' + rdatatype.to_text(covers) + ')'
        return '<DNS ' + DNS.rdataclass.to_text(self.rdclass) + ' ' + \
               DNS.rdatatype.to_text(self.rdtype) + ctext + ' rdata: ' + \
	       str(self) + '>'

    def __str__(self):
	return self.to_text()

    def __cmp__(self, other):
        if self.rdclass != other.rdclass or self.rdtype != other.rdtype:
	    raise MeaninglessComparison
        return 0

    def from_text(cls, rdclass, rdtype, tok, origin = None):
        """Build an rdata object from text format.

        @param rdclass: The rdata class
        @type rdclass: int
        @param rdtype: The rdata type
        @type rdtype: int
        @param tok: The tokenizer
        @type tok: DNS.tokenizer.Tokenizer
        @param origin: The origin to use for relative names
        @type origin: DNS.name.Name
        @rtype: DNS.rdata.Rdata instance"""

        raise NotImplementedError

    from_text = classmethod(from_text)

    def from_wire(cls, rdclass, rdtype, wire, current, rdlen, origin = None):
        """Build an rdata object from wire format
        
        @param rdclass: The rdata class
        @type rdclass: int
        @param rdtype: The rdata type
        @type rdtype: int
        @param wire: The wire-format message
        @type wire: string
        @param current: The offet in wire of the beginning of the rdata.
        @type current: int
        @param rdlen: The length of the wire-format rdata
        @type rdlen: int
        @param origin: The origin to use for relative names
        @type origin: DNS.name.Name
        @rtype: DNS.rdata.Rdata instance"""

        raise NotImplementedError

    from_wire = classmethod(from_wire)
    
     
class GenericRdata(Rdata):
    """Generate Rdata Class

    This class is used for rdata types for which we have no better
    implementation.  It implements the DNS "unknown RRs" scheme."""
    
    def __init__(self, rdclass, rdtype, data):
        super(GenericRdata, self).__init__(rdclass, rdtype)
        self.data = data
        
    def to_text(self, **kw):
        return r'\# %d ' % len(self.data) + _hexify(self.data)

    def from_text(cls, rdclass, rdtype, tok, origin = None):
        if tok.get() != r'\#':
            raise SyntaxError, r'generic rdata does not start with \#'
        length_text = tok.get()
        if not length_text.isdigit():
            raise SyntaxError, 'generic rdata length not a decimal number'
        length = int(length_text)
        chunks = []
        while 1:
            t = tok.get()
            if t == '\n' or t == '':
                break
            chunks.append(t)
        hex = ''.join(chunks)
        data = hex.decode('hex_codec')
        if len(data) != length:
            raise SyntaxError, 'generic rdata hex data has wrong length'
        return cls(rdclass, rdtype, data)

    from_text = classmethod(from_text)

    def to_wire(self, file, compress = None, origin = None):
        file.write(self.data)
        
    def from_wire(cls, rdclass, rdtype, wire, current, rdlen, origin = None):
        return cls(rdclass, rdtype, wire[current : current + rdlen])

    from_wire = classmethod(from_wire)

    def __cmp__(self, other):
        if self.rdclass != other.rdclass or self.rdtype != other.rdtype:
	    raise MeaninglessComparison
	return cmp(self.data, other.data)

_rdata_modules = {}
_module_prefix = 'DNS.rdtypes'

def get_rdata_class(rdclass, rdtype):

    def import_module(name):
        mod = __import__(name)
        components = name.split('.')
        for comp in components[1:]:
            mod = getattr(mod, comp)
        return mod
    
    mod = _rdata_modules.get((rdclass, rdtype))
    rdclass_text = DNS.rdataclass.to_text(rdclass)
    rdtype_text = DNS.rdatatype.to_text(rdtype)
    if not mod:
        mod = _rdata_modules.get((DNS.rdatatype.ANY, rdtype))
        if not mod:
            try:
                mod = import_module('.'.join([_module_prefix,
                                              rdclass_text, rdtype_text]))
                _rdata_modules[(rdclass, rdtype)] = mod
            except ImportError:
                try:
                    mod = import_module('.'.join([_module_prefix,
                                                  'ANY', rdtype_text]))
                    _rdata_modules[(DNS.rdataclass.ANY, rdtype)] = mod
                except ImportError:
                    mod = None
    if mod:
        cls = getattr(mod, rdtype_text)
    else:
        cls = GenericRdata
    return cls

def from_text(rdclass, rdtype, tok, origin = None):
    """Build an rdata object from text format.

    This function attempts to dynamically load a class which
    implements the specified rdata class and type.  If there is no
    class-and-type-specific implementation, the GenericRdata class
    is used.

    Once a class is chosen, its from_text() class method is called
    with the parameters to this function.

    @param rdclass: The rdata class
    @type rdclass: int
    @param rdtype: The rdata type
    @type rdtype: int
    @param tok: The tokenizer
    @type tok: DNS.tokenizer.Tokenizer
    @param origin: The origin to use for relative names
    @type origin: DNS.name.Name
    @rtype: DNS.rdata.Rdata instance"""
    
    if isinstance(tok, str):
        tok = DNS.tokenizer.Tokenizer(tok)
    cls = get_rdata_class(rdclass, rdtype)
    return cls.from_text(rdclass, rdtype, tok, origin)

def from_wire(rdclass, rdtype, wire, current, rdlen, origin = None):
    """Build an rdata object from wire format

    This function attempts to dynamically load a class which
    implements the specified rdata class and type.  If there is no
    class-and-type-specific implementation, the GenericRdata class
    is used.

    Once a class is chosen, its from_wire() class method is called
    with the parameters to this function.
    
    @param rdclass: The rdata class
    @type rdclass: int
    @param rdtype: The rdata type
    @type rdtype: int
    @param wire: The wire-format message
    @type wire: string
    @param current: The offet in wire of the beginning of the rdata.
    @type current: int
    @param rdlen: The length of the wire-format rdata
    @type rdlen: int
    @param origin: The origin to use for relative names
    @type origin: DNS.name.Name
    @rtype: DNS.rdata.Rdata instance"""

    cls = get_rdata_class(rdclass, rdtype)
    return cls.from_wire(rdclass, rdtype, wire, current, rdlen, origin)
