# Copyright (C) 2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: SRV.py,v 1.2 2003/05/29 00:31:40 halley Exp $

import struct

import DNS.rdata
import DNS.name

class SRV(DNS.rdata.Rdata):
    def __init__(self, rdclass, rdtype, priority, weight, port, target):
        super(SRV, self).__init__(rdclass, rdtype)
        self.priority = priority
        self.weight = weight
        self.port = port
        self.target = target

    def to_text(self, origin=None, relativize=True, **kw):
        target = self.target.choose_relativity(origin, relativize)
        return '%d %d %d %s' % (self.priority, self.weight, self.port,
                                target)
        
    def from_text(cls, rdclass, rdtype, tok, origin = None):
        priority = tok.get_uint16()
        weight = tok.get_uint16()
        port = tok.get_uint16()
        target = tok.get_name(None)
        if not origin is None:
            target = target.relativize(origin)
        tok.get_eol()
        return cls(rdclass, rdtype, priority, weight, port, target)
    
    from_text = classmethod(from_text)

    def to_wire(self, file, compress = None, origin = None):
        three_ints = struct.pack("!HHH", self.priority, self.weight, self.port)
        file.write(three_ints)
        self.target.to_wire(file, compress, origin)
        
    def from_wire(cls, rdclass, rdtype, wire, current, rdlen, origin = None):
        (priority, weight, port) = struct.unpack('!HHH',
                                                 wire[current : current + 6])
        current += 6
        rdlen -= 6
        (target, cused) = DNS.name.from_wire(wire[: current + rdlen],
                                             current)
        if cused != rdlen:
            raise SyntaxError
        if not origin is None:
            target = target.relativize(origin)
        return cls(rdclass, rdtype, priority, weight, port, target)

    from_wire = classmethod(from_wire)

    def __cmp__(self, other):
        sp = struct.pack("!HHH", self.priority, self.weight, self.port)
        op = struct.pack("!HHH", other.priority, self.weight, self.port)
        v = cmp(sp, op)
        if v == 0:
            v = cmp(self.target, other.target)
        return v
