# Copyright (C) 2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: ipv6.py,v 1.3 2003/06/21 10:58:53 halley Exp $

"""IPv6 helper functions."""

import re

import DNS.exception

v4_ending = re.compile(r'(.*):(\d+)\.(\d+)\.(\d+)\.(\d+)$')
colon_colon_start = re.compile(r'::.*')
colon_colon_end = re.compile(r'.*::$')

def inet_ntoa(address):
    """Convert a network format IPv6 address into text."""

    if len(address) != 16:
        raise DNS.exception.SyntaxError
    hex = address.encode('hex_codec').upper()
    chunks = []
    i = 0
    l = len(hex)
    while i < l:
        chunks.append(hex[i : i + 4])
        i += 4
    hex = ':'.join(chunks)
    return hex

def inet_aton(text):
    """Convert a text format IPv6 address into network format."""
    
    #
    # Our aim here is not something fast; we just want something that works.
    #

    if text == '::':
        text = '0::'
    #
    # Get rid of the icky dot-quad syntax if we have it.
    #
    m = v4_ending.match(text)
    if not m is None:
        text = "%s:%04x:%04x" % (m.group(1),
                                 int(m.group(2)) * 256 + int(m.group(3)),
                                 int(m.group(4)) * 256 + int(m.group(5)))
    #
    # Try to turn '::<whatever>' into ':<whatever>'; if no match try to
    # turn '<whatever>::' into '<whatever>:'
    #
    m = colon_colon_start.match(text)
    if not m is None:
        text = text[1:]
    else:
        m = colon_colon_end.match(text)
        if not m is None:
            text = text[:-1]
    #
    # Now canonicalize into 8 chunks of 4 hex digits each
    #

    chunks = text.split(':')
    l = len(chunks)
    if l > 8:
        raise DNS.exception.SyntaxError
    seen_empty = False
    canonical = []
    for c in chunks:
        if c == '':
            if seen_empty:
                raise DNS.exception.SyntaxError
            seen_empty = True
            for i in xrange(0, 8 - l + 1):
                canonical.append('0000')
        else:
            lc = len(c)
            if lc > 4:
                raise DNS.exception.SyntaxError
            if lc != 4:
                c = ('0' * (4 - lc)) + c
            canonical.append(c)
    if l < 8 and not seen_empty:
        raise DNS.exception.SyntaxError
    text = ''.join(canonical)

    #
    # Finally we can encode.
    #
    return text.decode('hex_codec')
