# Copyright (C) 2001-2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: rcode.py,v 1.5 2003/06/20 07:12:38 halley Exp $

"""DNS Result Codes."""

import DNS.exception

NOERROR = 0
FORMERR = 1
SERVFAIL = 2
NXDOMAIN = 3
NOTIMP = 4
REFUSED = 5
YXDOMAIN = 6
YXRRSET = 7
NXRRSET = 8
NOTAUTH = 9
NOTZONE = 10
BADVERS = 16

_by_text = {
    'NOERROR' : NOERROR,
    'FORMERR' : FORMERR,
    'SERVFAIL' : SERVFAIL,
    'NXDOMAIN' : NXDOMAIN,
    'NOTIMP' : NOTIMP,
    'REFUSED' : REFUSED,
    'YXDOMAIN' : YXDOMAIN,
    'YXRRSET' : YXRRSET,
    'NXRRSET' : NXRRSET,
    'NOTAUTH' : NOTAUTH,
    'NOTZONE' : NOTZONE,
    'BADVERS' : BADVERS
}

_by_value = {
    NOERROR : 'NOERROR',
    FORMERR : 'FORMERR',
    SERVFAIL : 'SERVFAIL',
    NXDOMAIN : 'NXDOMAIN',
    NOTIMP : 'NOTIMP',
    REFUSED : 'REFUSED',
    YXDOMAIN : 'YXDOMAIN',
    YXRRSET : 'YXRRSET',
    NXRRSET : 'NXRRSET',
    NOTAUTH : 'NOTAUTH',
    NOTZONE : 'NOTZONE',
    BADVERS : 'BADVERS'
}

class UnknownRcode(DNS.exception.DNSException):
    """Raised if an rcode is unknown."""
    pass

def from_text(text):
    """Convert text into an rcode.
    @rtype: string"""
    
    try:
        v = _by_text[text.upper()]
    except KeyError:
        raise UnknownRcode
    return v

def from_flags(flags, ednsflags):
    """Return the rcode value encoded by flags and ednsflags.
    @rtype: int"""
    
    return (flags & 0x000f) | ((ednsflags >> 20) & 0xff0)

def to_flags(value):
    """Return a (flags, ednsflags) tuple which encodes the rcode.
    @rtype: (int, int) tuple"""
    
    v = value & 0xf
    ev = (value & 0xff0) << 20
    return (v, ev)

def to_text(value):
    """Convert rcode into text.
    @rtype: string"""
    
    try:
        text = _by_value[value]
    except KeyError:
        raise UnknownRcode
    return text
