# Copyright (C) 2001-2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: rdataset.py,v 1.9 2003/06/20 07:12:38 halley Exp $

"""DNS rdatasets (an rdataset is a set of rdatas of a given type and class)"""

import random
import StringIO
import struct
import sys

import DNS.exception
import DNS.rdatatype
import DNS.rdataclass
import DNS.rdata

class SimpleSet(object):
    """A very simple set class.

    Sets are not in Python until 2.3, and we also don't have a hash
    function for rdata which deals with name equivalence, so if we
    use a set based on dictionaries we could end up with a set
    which contained objects that were equal.  To avoid all of this
    trouble, we have our own simple set class.  It implements a
    subset of the 2.3 Set interface using a list.

    @ivar items: A list of the items which are in the set
    @type items: list"""
    
    def __init__(self):
        """Initialize the set, making it empty."""
        self.items = []
        
    def add(self, item):
        """Add an item to the set."""
        if not item in self.items:
            self.items.append(item)

    def update(self, other):
        """Update the set, adding any elements from other which are not
        already in the set.
        @param other: the collection of items with which to update the set
        @type other: any iterable type"""
        for item in other.items:
            self.add(item)

    def clear(self):
        """Make the set empty."""
        self.items = []

    def __eq__(self, other):
        # Yes, this is inefficient
        for item in self.items:
            if not item in other.items:
                return False
        for item in other.items:
            if not item in self.items:
                return False
        return True
    
    def __ne__(self, other):
        return not self.__eq__(other)

    def __len__(self):
        return len(self.items)

    def __iter__(self):
        return iter(self.items)

class DifferingCovers(DNS.exception.DNSException):
    """Raised if an attempt is made to add a SIG whose covered type is not
    the same as that of the other rdatas in the rdataset."""
    pass

class IncompatibleTypes(DNS.exception.DNSException):
    """Raised if an attempt is made to add rdata of an incompatible type."""
    pass

class Rdataset(SimpleSet):
    """A DNS rdataset.

    @ivar rdclass: The class of the rdataset
    @type rdclass: int
    @ivar rdtype: The type of the rdataset
    @type rdtype: int
    @ivar covers: The covered type.  Usually this value is DNS.rdatatype.NONE,
    but if the rdtype is DNS.rdatatype.SIG, then the covers value will be
    the rdata type the SIG covers.  The library treats the SIG type as if
    it were a family of types, e.g. SIG(A), SIG(NS), SIG(SOA).  This makes
    SIGs much easier to work with than if SIGs covering different rdata types
    were aggregated into a single SIG rdataset.
    @type covers: int
    @ivar ttl: The DNS TTL (Time To Live) value
    @type ttl: int
    @ivar deleting: If non-zero, the value of deleting should be the
    class to use when converting the rdataset to text or wire format.
    This field is used in dynamic update operations, for example in the
    "Delete an RR from an RRset" case, the deleting value will be
    DNS.rdatatype.NONE.
    @type deleting: int"""

    def __init__(self, rdclass, rdtype, deleting=0):
        """Create a new rdataset of the specified class and type.

        @see: the description of the class instance variables for the
        meaning of I{rdclass}, I{rdtype}, and I{deleting}."""
        
        super(Rdataset, self).__init__()
        self.rdclass = rdclass
        self.rdtype = rdtype
        self.covers = DNS.rdatatype.NONE
        self.ttl = 0
        self.deleting = deleting

    def update_ttl(self, ttl):
        """Set the TTL of the rdataset to be the lesser of the set's current
        TTL or the specified TTL.  If the set contains no rdatas, set the TTL
        to the specified TTL.
        @param ttl: The TTL
        @type ttl: int"""
        
        if len(self) == 0:
            self.ttl = ttl
        elif ttl < self.ttl:
            self.ttl = ttl

    def add(self, rd, ttl=None):
        """Add the specified rdata to the rdataset.

        If the optional I{ttl} parameter is supplied, then
        self.update_ttl(ttl) will be called prior to adding the rdata.
        
        @param rd: The rdata
        @type rd: DNS.rdata.Rdata object
        @param ttl: The TTL
        @type ttl: int"""
        
        #
        # If we're adding a signature, do some special handling to
        # check that the signature covers the same type as the
        # other rdatas in this rdataset.  If this is the first rdata
        # in the set, initialize the covers field.
        #
        if self.rdclass != rd.rdclass or self.rdtype != rd.rdtype:
            raise IncompatibleTypes
        if not ttl is None:
            self.update_ttl(ttl)
        if self.rdtype == DNS.rdatatype.SIG:
            covers = rd.covers()
            if len(self) == 0:
                self.covers = covers
            elif self.covers != covers:
                raise DifferingCovers
        if DNS.rdatatype.is_singleton(rd.rdtype) and len(self) > 0:
            self.clear()
        super(Rdataset, self).add(rd)
        
    def update(self, other):
        """Add all rdatas in other to self.

        @param other: The rdataset from which to update
        @type other: DNS.rdataset.Rdataset object"""
        
        self.update_ttl(other.ttl)
        super(Rdataset, self).update(other)

    def __repr__(self):
        if self.covers == 0:
            ctext = ''
        else:
            ctext = '(' + DNS.rdatatype.to_text(self.covers) + ')'
        return '<DNS ' + DNS.rdataclass.to_text(self.rdclass) + ' ' + \
               DNS.rdatatype.to_text(self.rdtype) + ctext + ' rdataset>'
    
    def __str__(self):
        return self.to_text()

    def __eq__(self, other):
        """Two rdatasets are equal if they have the same class, type, covers,
        deletion state, and contain the same rdata.
        @rtype: bool"""
        if self.rdclass != other.rdclass or \
           self.rdtype != other.rdtype or \
           self.covers != other.covers or \
           self.deleting != other.deleting:
            return False
        return super(Rdataset, self).__eq__(other)
    
    def __ne__(self, other):
        return not self.__eq__(other)

    def to_text(self, name=None, origin=None, relativize=True, **kw):
        """Convert the rdataset into DNS master file format.

        @see: L{DNS.name.Name.choose_relativity} for more information
        on how I{origin} and I{relativize} determine the way names
        are emitted.

        Any additional keyword arguments are passed on to the rdata
        to_text() method.
        
        @param name: If name is not None, emit a RRs with I{name} as
        the owner name.
        @type name: DNS.name.Name object
        @param origin: The origin for relative names, or None.
        @type origin: DNS.name.Name object
        @param relativize: True if names should names be relativized
        @type relativize: bool"""
        if not name is None:
            name = name.choose_relativity(origin, relativize)
            ntext = str(name)
            pad = ' '
        else:
            ntext = ''
            pad = ''
        s = StringIO.StringIO()
        if len(self) == 0:
            #
            # Empty rdatasets are used for the question section, and in
            # some dynamic updates, so we don't need to print out the TTL
            # (which is meaningless anyway).
            #
            if self.deleting:
                rdclass = self.deleting
            else:
                rdclass = self.rdclass
            print >> s, '%s%s%s %s' % (ntext, pad,
                                       DNS.rdataclass.to_text(rdclass),
                                       DNS.rdatatype.to_text(self.rdtype))
        else:
            if self.deleting:
                rdclass = DNS.rdataclass.NONE
            else:
                rdclass = self.rdclass
            for rd in self:
                print >> s, '%s%s%d %s %s %s' % \
                      (ntext, pad, self.ttl, DNS.rdataclass.to_text(rdclass),
                       DNS.rdatatype.to_text(self.rdtype),
                       rd.to_text(origin=origin, relativize=relativize, **kw))
        #
        # We strip off the final \n for the caller's convenience in printing
        #
        return s.getvalue()[:-1]

    def to_wire(self, name, file, compress=None, origin=None, question=False):
        """Convert the rdataset to wire format.

        @param name: The owner name of the RRset that will be emitted
        @type name: DNS.name.Name object
        @param file: The file to which the wire format data will be appended
        @type file: file
        @param compress: The compression table to use; the default is None.
        @type compress: dict
        @param origin: The origin to be appended to any relative names when
        they are emitted.  The default is None.
        @param question: If True, render this rdataset in the format of
        the question section, i.e. emit only the owner name, rdata class,
        and rdata type.
        @type question: bool
        @returns: the number of records emitted
        @rtype: int"""
        
        if self.deleting:
            rdclass = self.deleting
        else:
            rdclass = self.rdclass
        if len(self) == 0:
            file.seek(0, 2)
            name.to_wire(file, compress, origin)
            if question:
                stuff = struct.pack("!HH", self.rdtype, rdclass)
            else:
                stuff = struct.pack("!HHIH", self.rdtype, rdclass, 0, 0)
            file.write(stuff)
            return 1
        else:
            if self.deleting:
                l = self
            else:
                l = list(self)
                random.shuffle(l)
            for rd in l:
                file.seek(0, 2)
                name.to_wire(file, compress, origin)
                stuff = struct.pack("!HHIH", self.rdtype, rdclass,
                                    self.ttl, 0)
                file.write(stuff)
                start = file.tell()
                rd.to_wire(file, compress, origin)
                end = file.tell()
                assert end - start < 65536
                file.seek(start - 2, 0)
                stuff = struct.pack("!H", end - start)
                file.write(stuff)
            return len(self)
    
    def match(self, rdclass, rdtype, covers, deleting=0):
        """Returns True if this rdataset matches the specified class, type,
        covers and optional deletion state."""
        if self.rdclass == rdclass and \
           self.rdtype == rdtype and \
           self.covers == covers and \
           self.deleting == deleting:
            return True
        return False

def from_text_list(rdclass, rdtype, ttl, text_rdatas):
    """Create an rdataset of the specified class, type, and TTL, and with
    the specified list of rdatas in text format."""
    r = Rdataset(rdclass, rdtype)
    r.update_ttl(ttl)
    for t in text_rdatas:
        rd = DNS.rdata.from_text(r.rdclass, r.rdtype, t)
        r.add(rd)
    return r
    
def from_text(rdclass, rdtype, ttl, *text_rdatas):
    """Create an rdataset of the specified class, type, and TTL, and with
    the specified rdatas in text format."""
    return from_text_list(rdclass, rdtype, ttl, text_rdatas)

def from_rdata_list(ttl, rdatas):
    """Create an rdataset of the specified class, type, and TTL, and with
    the specified list of rdata objects."""
    assert len(rdatas) > 0
    first_time = True
    for rd in rdatas:
        if first_time:
            r = Rdataset(rd.rdclass, rd.rdtype)
            r.update_ttl(ttl)
            first_time = False
        r.add(rd)
    return r
    
def from_rdata(ttl, *rdatas):
    """Create an rdataset of the specified class, type, and TTL, and with
    the specified rdata objects."""
    return from_rdata_list(ttl, rdatas)
