# Copyright (C) 2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: NXT.py,v 1.4 2003/06/20 07:12:38 halley Exp $

import struct

import DNS.exception
import DNS.rdata
import DNS.rdatatype
import DNS.name

class NXT(DNS.rdata.Rdata):
    """NXT record

    @ivar next: the next name
    @type next: DNS.name.Name object
    @ivar bitmap: the type bitmap
    @type bitmap: string
    @see: RFC 2535"""

    def __init__(self, rdclass, rdtype, next, bitmap):
        super(NXT, self).__init__(rdclass, rdtype)
        self.next = next
        self.bitmap = bitmap

    def to_text(self, origin=None, relativize=True, **kw):
        next = self.next.choose_relativity(origin, relativize)
        bits = []
        for i in xrange(0, len(self.bitmap)):
            byte = ord(self.bitmap[i])
            for j in xrange(0, 8):
                if byte & (0x80 >> j):
                    bits.append(DNS.rdatatype.to_text(i * 8 + j))
        text = ' '.join(bits)
        return '%s %s' % (next, text)
        
    def from_text(cls, rdclass, rdtype, tok, origin = None):
        next = tok.get_name(None)
        bitmap = ['\x00', '\x00', '\x00', '\x00',
                  '\x00', '\x00', '\x00', '\x00',
                  '\x00', '\x00', '\x00', '\x00',
                  '\x00', '\x00', '\x00', '\x00' ]
        while 1:
            (ttype, value) = tok.get()
            if ttype == DNS.tokenizer.EOL or ttype == DNS.tokenizer.EOF:
                break
            if value.isdigit():
                nrdtype = int(value)
            else:
                nrdtype = DNS.rdatatype.from_text(value)
            if nrdtype == 0:
                raise DNS.exception.SyntaxError, "NXT with bit 0"
            if nrdtype > 127:
                raise DNS.exception.SyntaxError, "NXT with bit > 127"
            i = nrdtype // 8
            bitmap[i] = chr(ord(bitmap[i]) | (0x80 >> (nrdtype % 8)))
        bitmap = DNS.rdata._truncate_bitmap(bitmap)
        return cls(rdclass, rdtype, next, bitmap)
    
    from_text = classmethod(from_text)

    def to_wire(self, file, compress = None, origin = None):
        self.next.to_wire(file, None, origin)
        file.write(self.bitmap)
        
    def from_wire(cls, rdclass, rdtype, wire, current, rdlen, origin = None):
        (next, cused) = DNS.name.from_wire(wire[: current + rdlen], current)
        current += cused
        rdlen -= cused
        bitmap = wire[current : current + rdlen]
        if not origin is None:
            next = next.relativize(origin)
        return cls(rdclass, rdtype, next, bitmap)

    from_wire = classmethod(from_wire)

    def __cmp__(self, other):
        v = cmp(self.next, other.next)
        if v == 0:
            v = cmp(self.bitmap, other.bitmap)
        return v
