# Copyright (C) 2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: RP.py,v 1.5 2003/06/20 07:12:38 halley Exp $

import struct

import DNS.exception
import DNS.rdata
import DNS.name

class RP(DNS.rdata.Rdata):
    """RP record

    @ivar mbox: The responsible person's mailbox
    @type mbox: DNS.name.Name object
    @ivar txt: The owner name of a node with TXT records, or the root name
    if no TXT records are associated with this RP.
    @type txt: DNS.name.Name object
    @see: RFC 1183"""
    
    def __init__(self, rdclass, rdtype, mbox, txt):
        super(RP, self).__init__(rdclass, rdtype)
        self.mbox = mbox
        self.txt = txt

    def to_text(self, origin=None, relativize=True, **kw):
        mbox = self.mbox.choose_relativity(origin, relativize)
        txt = self.txt.choose_relativity(origin, relativize)
        return "%s %s" % (str(mbox), str(txt))
        
    def from_text(cls, rdclass, rdtype, tok, origin = None):
        mbox = tok.get_name(None)
        if not origin is None:
            mbox = mbox.relativize(origin)
        txt = tok.get_name(None)
        if not origin is None:
            txt = txt.relativize(origin)
        tok.get_eol()
        return cls(rdclass, rdtype, mbox, txt)
    
    from_text = classmethod(from_text)

    def to_wire(self, file, compress = None, origin = None):
        self.mbox.to_wire(file, None, origin)
        self.txt.to_wire(file, None, origin)
        
    def from_wire(cls, rdclass, rdtype, wire, current, rdlen, origin = None):
        (mbox, cused) = DNS.name.from_wire(wire[: current + rdlen],
                                           current)
        current += cused
        rdlen -= cused
        if rdlen <= 0:
            raise DNS.exception.FormError
        (txt, cused) = DNS.name.from_wire(wire[: current + rdlen],
                                          current)
        if cused != rdlen:
            raise DNS.exception.FormError
        if not origin is None:
            mbox = mbox.relativize(origin)
            txt = txt.relativize(origin)
        return cls(rdclass, rdtype, mbox, txt)

    from_wire = classmethod(from_wire)

    def __cmp__(self, other):
        v = cmp(self.mbox, other.mbox)
        if v == 0:
            v = cmp(self.txt, other.txt)
        return v
