# Copyright (C) 2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: PX.py,v 1.2 2003/06/20 07:12:38 halley Exp $

import struct

import DNS.exception
import DNS.rdata
import DNS.name

class PX(DNS.rdata.Rdata):
    """PX record.

    @ivar preference: the preference value
    @type preference: int
    @ivar map822: the map822 name
    @type map822: DNS.name.Name object
    @ivar mapx400: the mapx400 name
    @type mapx400: DNS.name.Name object
    @see: RFC 2163"""
    
    def __init__(self, rdclass, rdtype, preference, map822, mapx400):
        super(PX, self).__init__(rdclass, rdtype)
        self.preference = preference
        self.map822 = map822
        self.mapx400 = mapx400

    def to_text(self, origin=None, relativize=True, **kw):
        map822 = self.map822.choose_relativity(origin, relativize)
        mapx400 = self.mapx400.choose_relativity(origin, relativize)
        return '%d %s %s' % (self.preference, map822, mapx400)
        
    def from_text(cls, rdclass, rdtype, tok, origin = None):
        preference = tok.get_uint16()
        map822 = tok.get_name(None)
        if not origin is None:
            map822 = map822.relativize(origin)
        mapx400 = tok.get_name(None)
        if not origin is None:
            mapx400 = mapx400.relativize(origin)
        tok.get_eol()
        return cls(rdclass, rdtype, preference, map822, mapx400)
    
    from_text = classmethod(from_text)

    def to_wire(self, file, compress = None, origin = None):
        pref = struct.pack("!H", self.preference)
        file.write(pref)
        self.map822.to_wire(file, None, origin)
        self.mapx400.to_wire(file, None, origin)
        
    def from_wire(cls, rdclass, rdtype, wire, current, rdlen, origin = None):
        (preference, ) = struct.unpack('!H', wire[current : current + 2])
        current += 2
        rdlen -= 2
        (map822, cused) = DNS.name.from_wire(wire[: current + rdlen],
                                               current)
        if cused > rdlen:
            raise DNS.exception.FormError
        current += cused
        rdlen -= cused
        if not origin is None:
            map822 = map822.relativize(origin)
        (mapx400, cused) = DNS.name.from_wire(wire[: current + rdlen],
                                              current)
        if cused != rdlen:
            raise DNS.exception.FormError
        if not origin is None:
            mapx400 = mapx400.relativize(origin)
        return cls(rdclass, rdtype, preference, map822, mapx400)

    from_wire = classmethod(from_wire)

    def __cmp__(self, other):
        sp = struct.pack("!H", self.preference)
        op = struct.pack("!H", other.preference)
        v = cmp(sp, op)
        if v == 0:
            v = cmp(self.map822, other.map822)
            if v == 0:
                v = cmp(self.mapx400, other.mapx400)
        return v
