# Copyright (C) 2003 Nominum, Inc.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose with or without fee is hereby granted,
# provided that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND NOMINUM DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL NOMINUM BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: WKS.py,v 1.2 2003/06/21 10:58:53 halley Exp $

import socket
import struct

import DNS.ipv4
import DNS.rdata

_proto_tcp = socket.getprotobyname('tcp')
_proto_udp = socket.getprotobyname('udp')

class WKS(DNS.rdata.Rdata):
    """WKS record

    @ivar address: the address
    @type address: string
    @ivar protocol: the protocol
    @type protocol: int
    @ivar bitmap: the bitmap
    @type bitmap: string
    @see: RFC 1035"""

    def __init__(self, rdclass, rdtype, address, protocol, bitmap):
        super(WKS, self).__init__(rdclass, rdtype)
        self.address = address
        self.protocol = protocol
        self.bitmap = bitmap

    def to_text(self, origin=None, relativize=True, **kw):
        bits = []
        for i in xrange(0, len(self.bitmap)):
            byte = ord(self.bitmap[i])
            for j in xrange(0, 8):
                if byte & (0x80 >> j):
                    bits.append(str(i * 8 + j))
        text = ' '.join(bits)
        return '%s %d %s' % (self.address, self.protocol, text)
        
    def from_text(cls, rdclass, rdtype, tok, origin = None):
        address = tok.get_string()
        protocol = tok.get_string()
        if protocol.isdigit():
            protocol = int(protocol)
        else:
            protocol = socket.getprotobyname(protocol)
        bitmap = []
        while 1:
            (ttype, value) = tok.get()
            if ttype == DNS.tokenizer.EOL or ttype == DNS.tokenizer.EOF:
                break
            if value.isdigit():
                serv = int(value)
            else:
                if protocol != _proto_udp and protocol != _proto_tcp:
                    raise NotImplementedError, "protocol must be TCP or UDP"
                if protocol == _proto_udp:
                    protocol_text = "udp"
                else:
                    protocol_text = "tcp"
                serv = socket.getservbyname(value, protocol_text)
            i = serv // 8
            l = len(bitmap)
            if l < i + 1:
                for j in xrange(l, i + 1):
                    bitmap.append('\x00')
            bitmap[i] = chr(ord(bitmap[i]) | (0x80 >> (serv % 8)))
        bitmap = DNS.rdata._truncate_bitmap(bitmap)
        return cls(rdclass, rdtype, address, protocol, bitmap)
    
    from_text = classmethod(from_text)

    def to_wire(self, file, compress = None, origin = None):
        file.write(DNS.ipv4.inet_aton(self.address))
        protocol = struct.pack('!H', self.protocol)
        file.write(protocol)
        file.write(self.bitmap)
        
    def from_wire(cls, rdclass, rdtype, wire, current, rdlen, origin = None):
        address = DNS.ipv4.inet_ntoa(wire[current : current + 4])
        protocol, = struct.unpack('!H', wire[current + 4 : current + 6])
        current += 6
        rdlen -= 6
        bitmap = wire[current : current + rdlen]
        return cls(rdclass, rdtype, address, protocol, bitmap)

    from_wire = classmethod(from_wire)

    def __cmp__(self, other):
        sa = DNS.ipv4.inet_aton(self.address)
        oa = DNS.ipv4.inet_aton(other.address)
        v = cmp(sa, oa)
        if v == 0:
            sp = struct.pack('!H', self.protocol)
            op = struct.pack('!H', other.protocol)
            v = cmp(sp, op)
            if v == 0:
                v = cmp(self.bitmap, other.bitmap)
        return v
